<?php
// index.php
// Versión 5:
// - Login con usuarios en BD
// - Formulario de creación/edición de reporte
// - Bloquea duplicados por Franquicia + Mes + Año
// - Mantiene cálculo de indicadores, reporte y gráficas

require_once __DIR__ . '/auth.php';
require_login();

// Helper para leer números (permitiendo coma o punto)
function getNumber($name) {
    $v = $_POST[$name] ?? 0;
    $v = str_replace(['.', ','], ['', '.'], $v);
    return (float)$v;
}

// Helper para fecha bonita
function fechaBonita($fecha) {
    if (!$fecha) return '';
    try {
        $dt = new DateTime($fecha);
        return $dt->format('d/m/Y');
    } catch (Exception $e) {
        return htmlspecialchars($fecha);
    }
}

$datos = [];
$resultados = [];
$errores = [];

// Meses para el desplegable
$MESES_REPORTE = [
    'Enero', 'Febrero', 'Marzo', 'Abril', 'Mayo', 'Junio',
    'Julio', 'Agosto', 'Septiembre', 'Octubre', 'Noviembre', 'Diciembre'
];

$GUARDAR_EN_BD = true;
global $pdo;

// ID para editar (vía GET o POST)
$id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $id = (int)($_POST['id'] ?? 0);
}

// Si es edición (GET) y no hay POST, cargamos datos existentes
if ($_SERVER['REQUEST_METHOD'] !== 'POST' && $id > 0) {
    $stmt = $pdo->prepare("SELECT * FROM proyecciones WHERE id = :id");
    $stmt->execute([':id' => $id]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($row) {
        $datos = [
            'franquicia'            => $row['franquicia'],
            'responsable_nombre'    => $row['responsable_nombre'],
            'responsable_apellidos' => $row['responsable_apellidos'],
            'departamento'          => $row['departamento'],
            'ciudad'                => $row['ciudad'],
            'email'                 => $row['email'],
            'fecha_reporte'         => $row['fecha_reporte'],
            'mes_analisis'          => $row['mes_analisis'],
            'ventas_mes'            => (float)$row['ventas_mes'],
            'num_empleados'         => (int)$row['num_empleados'],
            'arriendo_admin'        => (float)$row['arriendo_admin'],
            'nomina'                => (float)$row['nomina'],
            'servicios_publicos'    => (float)$row['servicios_publicos'],
            'contador'              => (float)$row['contador'],
            'calidad'               => (float)$row['calidad'],
            'papeleria'             => (float)$row['papeleria'],
            'transporte_parqueadero'=> (float)$row['transporte_parqueadero'],
            'impuestos'             => (float)$row['impuestos'],
            'mantenimiento'         => (float)$row['mantenimiento'],
            'mercadeo_activaciones' => (float)$row['mercadeo_activaciones'],
            'plataformas'           => (float)$row['plataformas'],
            'aseo_insumos'          => (float)$row['aseo_insumos'],
            'otros_gastos'          => (float)$row['otros_gastos'],
            'credito'               => (float)$row['credito'],
            'gastos_legales'        => (float)$row['gastos_legales'],
            'gastos_financieros'    => (float)$row['gastos_financieros'],
            'cafe'                  => (float)$row['cafe'],
            'leche'                 => (float)$row['leche'],
            'frutas_verduras'       => (float)$row['frutas_verduras'],
            'reposteria'            => (float)$row['reposteria'],
            'otros_costos'          => (float)$row['otros_costos'],
            'inventario'            => (float)$row['inventario'],
        ];
    } else {
        $errores[] = 'No se encontró el reporte solicitado para editar.';
        $id = 0;
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // ===== Campos básicos =====
    $datos['franquicia']            = trim($_POST['franquicia'] ?? '');
    $datos['responsable_nombre']    = trim($_POST['responsable_nombre'] ?? '');
    $datos['responsable_apellidos'] = trim($_POST['responsable_apellidos'] ?? '');
    $datos['departamento']          = trim($_POST['departamento'] ?? '');
    $datos['ciudad']                = trim($_POST['ciudad'] ?? '');
    $datos['email']                 = trim($_POST['email'] ?? '');
    $datos['fecha_reporte']         = trim($_POST['fecha_reporte'] ?? '');
    $datos['mes_analisis']          = trim($_POST['mes_analisis'] ?? '');

    $datos['ventas_mes']    = getNumber('ventas_mes');
    $datos['num_empleados'] = (int)($_POST['num_empleados'] ?? 0);

    if ($datos['franquicia'] === '') $errores[] = 'El nombre de la franquicia es obligatorio.';
    if ($datos['responsable_nombre'] === '') $errores[] = 'El nombre del responsable es obligatorio.';
    if ($datos['departamento'] === '') $errores[] = 'El departamento es obligatorio.';
    if ($datos['ciudad'] === '') $errores[] = 'La ciudad es obligatoria.';
    if ($datos['email'] === '') $errores[] = 'El email es obligatorio.';
    if ($datos['fecha_reporte'] === '') $errores[] = 'La fecha de reporte es obligatoria.';
    if ($datos['mes_analisis'] === '') $errores[] = 'El mes de reporte es obligatorio.';
    if ($datos['ventas_mes'] <= 0) $errores[] = 'Las ventas del mes deben ser mayores a 0.';
    if ($datos['num_empleados'] <= 0) $errores[] = 'El número de empleados debe ser mayor a 0.';

    // ===== Validar duplicado (franquicia + mes + año) =====
    if (empty($errores)) {
        $sqlDup = "SELECT id FROM proyecciones
                   WHERE franquicia = :franquicia
                     AND mes_analisis = :mes
                     AND YEAR(fecha_reporte) = YEAR(:fecha)";
        $paramsDup = [
            ':franquicia' => $datos['franquicia'],
            ':mes'        => $datos['mes_analisis'],
            ':fecha'      => $datos['fecha_reporte'],
        ];
        if ($id > 0) {
            $sqlDup .= " AND id <> :id";
            $paramsDup[':id'] = $id;
        }
        $stmtDup = $pdo->prepare($sqlDup);
        $stmtDup->execute($paramsDup);
        if ($stmtDup->fetch()) {
            $errores[] = 'Ya existe un reporte para esta franquicia en ese mes y año. Por favor edítalo en lugar de crear uno nuevo.';
        }
    }

    // ===== Gastos =====
    $datos['arriendo_admin']         = getNumber('arriendo_admin');
    $datos['nomina']                 = getNumber('nomina');
    $datos['servicios_publicos']     = getNumber('servicios_publicos');
    $datos['contador']               = getNumber('contador');
    $datos['calidad']                = getNumber('calidad');
    $datos['papeleria']              = getNumber('papeleria');
    $datos['transporte_parqueadero'] = getNumber('transporte_parqueadero');
    $datos['impuestos']              = getNumber('impuestos');
    $datos['mantenimiento']          = getNumber('mantenimiento');
    $datos['mercadeo_activaciones']  = getNumber('mercadeo_activaciones');
    $datos['plataformas']            = getNumber('plataformas');
    $datos['aseo_insumos']           = getNumber('aseo_insumos');
    $datos['otros_gastos']           = getNumber('otros_gastos');
    $datos['credito']                = getNumber('credito');
    $datos['gastos_legales']         = getNumber('gastos_legales');
    $datos['gastos_financieros']     = getNumber('gastos_financieros');

    $resultados['total_gastos'] =
        $datos['arriendo_admin'] +
        $datos['nomina'] +
        $datos['servicios_publicos'] +
        $datos['contador'] +
        $datos['calidad'] +
        $datos['papeleria'] +
        $datos['transporte_parqueadero'] +
        $datos['mantenimiento'] +
        $datos['mercadeo_activaciones'] +
        $datos['plataformas'] +
        $datos['aseo_insumos'] +
        $datos['otros_gastos'] +
        $datos['gastos_legales'] +
        $datos['gastos_financieros'];

    // ===== Costos =====
    $datos['cafe']            = getNumber('cafe');
    $datos['leche']           = getNumber('leche');
    $datos['frutas_verduras'] = getNumber('frutas_verduras');
    $datos['reposteria']      = getNumber('reposteria');
    $datos['otros_costos']    = getNumber('otros_costos');

    $resultados['total_costos'] =
        $datos['cafe'] +
        $datos['leche'] +
        $datos['frutas_verduras'] +
        $datos['reposteria'] +
        $datos['otros_costos'];

    // ===== Inventario =====
    $datos['inventario'] = getNumber('inventario');
    $resultados['inventario_sugerido'] = $datos['ventas_mes'] / 10;

    // ===== Cálculos principales =====
    $resultados['promedio_dia'] = $datos['ventas_mes'] / 30;

    if ($datos['ventas_mes'] > 0) {
        $resultados['porcentaje_costos'] = ($resultados['total_costos'] / $datos['ventas_mes']) * 100;
    } else {
        $resultados['porcentaje_costos'] = 0;
    }

    $resultados['utilidad_bruta']        = ($datos['ventas_mes'] - $resultados['total_costos']) + $datos['inventario'];
    $resultados['utilidad_operacional']  = $resultados['utilidad_bruta'] - $resultados['total_gastos'];
    $resultados['utilidad_neta']         =
        $resultados['utilidad_operacional'] -
        $datos['credito'] -
        $datos['gastos_financieros'] -
        $datos['impuestos'];

    $resultados['utilidad_neta_menos_inventario'] =
        $resultados['utilidad_neta'] - $datos['inventario'];

    if ($datos['ventas_mes'] > 0) {
        $resultados['utilidad_neta_menos_inv_sobre_ventas'] =
            ($resultados['utilidad_neta_menos_inventario'] / $datos['ventas_mes']) * 100;
    } else {
        $resultados['utilidad_neta_menos_inv_sobre_ventas'] = 0;
    }

    $resultados['costo_produccion_pct'] = $resultados['porcentaje_costos'];

    if ($datos['ventas_mes'] > 0) {
        $resultados['part_arriendo_pct']      = ($datos['arriendo_admin']     / $datos['ventas_mes']) * 100;
        $resultados['part_servicios_pct']     = ($datos['servicios_publicos'] / $datos['ventas_mes']) * 100;
        $resultados['part_nomina_pct']        = ($datos['nomina']             / $datos['ventas_mes']) * 100;
        $resultados['margen_neto_pct']        = ($resultados['utilidad_neta'] / $datos['ventas_mes']) * 100;
        $resultados['margen_operacional_pct'] =
            ($resultados['utilidad_operacional'] / $datos['ventas_mes']) * 100;
    } else {
        $resultados['part_arriendo_pct'] =
        $resultados['part_servicios_pct'] =
        $resultados['part_nomina_pct'] =
        $resultados['margen_neto_pct'] =
        $resultados['margen_operacional_pct'] = 0;
    }

    if ($resultados['total_costos'] > 0) {
        $resultados['part_cafe_pct']         = ($datos['cafe']            / $resultados['total_costos']) * 100;
        $resultados['part_leche_pct']        = ($datos['leche']           / $resultados['total_costos']) * 100;
        $resultados['part_frutas_pct']       = ($datos['frutas_verduras'] / $resultados['total_costos']) * 100;
        $resultados['part_reposteria_pct']   = ($datos['reposteria']      / $resultados['total_costos']) * 100;
        $resultados['part_otros_costos_pct'] = ($datos['otros_costos']    / $resultados['total_costos']) * 100;
    } else {
        $resultados['part_cafe_pct'] =
        $resultados['part_leche_pct'] =
        $resultados['part_frutas_pct'] =
        $resultados['part_reposteria_pct'] =
        $resultados['part_otros_costos_pct'] = 0;
    }

    // ===== Guardar en base de datos =====
    if ($GUARDAR_EN_BD && empty($errores)) {
        try {
            if ($id > 0) {
                // UPDATE (editar)
                $sql = "UPDATE proyecciones SET
                    franquicia = :franquicia,
                    responsable_nombre = :responsable_nombre,
                    responsable_apellidos = :responsable_apellidos,
                    departamento = :departamento,
                    ciudad = :ciudad,
                    email = :email,
                    fecha_reporte = :fecha_reporte,
                    mes_analisis = :mes_analisis,
                    ventas_mes = :ventas_mes,
                    num_empleados = :num_empleados,
                    promedio_dia = :promedio_dia,
                    arriendo_admin = :arriendo_admin,
                    nomina = :nomina,
                    servicios_publicos = :servicios_publicos,
                    contador = :contador,
                    calidad = :calidad,
                    papeleria = :papeleria,
                    transporte_parqueadero = :transporte_parqueadero,
                    impuestos = :impuestos,
                    mantenimiento = :mantenimiento,
                    mercadeo_activaciones = :mercadeo_activaciones,
                    plataformas = :plataformas,
                    aseo_insumos = :aseo_insumos,
                    otros_gastos = :otros_gastos,
                    credito = :credito,
                    gastos_legales = :gastos_legales,
                    gastos_financieros = :gastos_financieros,
                    total_gastos = :total_gastos,
                    cafe = :cafe,
                    leche = :leche,
                    frutas_verduras = :frutas_verduras,
                    reposteria = :reposteria,
                    otros_costos = :otros_costos,
                    total_costos = :total_costos,
                    porcentaje_costos = :porcentaje_costos,
                    inventario = :inventario,
                    inventario_sugerido = :inventario_sugerido,
                    utilidad_bruta = :utilidad_bruta,
                    utilidad_operacional = :utilidad_operacional,
                    utilidad_neta = :utilidad_neta,
                    utilidad_neta_menos_inventario = :utilidad_neta_menos_inventario,
                    utilidad_neta_menos_inv_sobre_ventas = :utilidad_neta_menos_inv_sobre_ventas,
                    costo_produccion_pct = :costo_produccion_pct,
                    part_arriendo_pct = :part_arriendo_pct,
                    part_servicios_pct = :part_servicios_pct,
                    part_nomina_pct = :part_nomina_pct,
                    margen_neto_pct = :margen_neto_pct,
                    margen_operacional_pct = :margen_operacional_pct,
                    part_cafe_pct = :part_cafe_pct,
                    part_leche_pct = :part_leche_pct,
                    part_frutas_pct = :part_frutas_pct,
                    part_reposteria_pct = :part_reposteria_pct,
                    part_otros_costos_pct = :part_otros_costos_pct
                  WHERE id = :id";
                $stmt = $pdo->prepare($sql);
                $stmt->execute(array_merge($datos, $resultados, ['id' => $id]));
            } else {
                // INSERT (nuevo)
                $sql = "INSERT INTO proyecciones (
                    franquicia, responsable_nombre, responsable_apellidos,
                    departamento, ciudad, email, fecha_reporte, mes_analisis,
                    ventas_mes, num_empleados, promedio_dia,
                    arriendo_admin, nomina, servicios_publicos, contador, calidad, papeleria,
                    transporte_parqueadero, impuestos, mantenimiento, mercadeo_activaciones,
                    plataformas, aseo_insumos, otros_gastos, credito, gastos_legales,
                    gastos_financieros, total_gastos,
                    cafe, leche, frutas_verduras, reposteria, otros_costos, total_costos,
                    porcentaje_costos, inventario, inventario_sugerido,
                    utilidad_bruta, utilidad_operacional, utilidad_neta,
                    utilidad_neta_menos_inventario, utilidad_neta_menos_inv_sobre_ventas,
                    costo_produccion_pct, part_arriendo_pct, part_servicios_pct,
                    part_nomina_pct, margen_neto_pct, margen_operacional_pct,
                    part_cafe_pct, part_leche_pct, part_frutas_pct,
                    part_reposteria_pct, part_otros_costos_pct
                  ) VALUES (
                    :franquicia, :responsable_nombre, :responsable_apellidos,
                    :departamento, :ciudad, :email, :fecha_reporte, :mes_analisis,
                    :ventas_mes, :num_empleados, :promedio_dia,
                    :arriendo_admin, :nomina, :servicios_publicos, :contador, :calidad, :papeleria,
                    :transporte_parqueadero, :impuestos, :mantenimiento, :mercadeo_activaciones,
                    :plataformas, :aseo_insumos, :otros_gastos, :credito, :gastos_legales,
                    :gastos_financieros, :total_gastos,
                    :cafe, :leche, :frutas_verduras, :reposteria, :otros_costos, :total_costos,
                    :porcentaje_costos, :inventario, :inventario_sugerido,
                    :utilidad_bruta, :utilidad_operacional, :utilidad_neta,
                    :utilidad_neta_menos_inventario, :utilidad_neta_menos_inv_sobre_ventas,
                    :costo_produccion_pct, :part_arriendo_pct, :part_servicios_pct,
                    :part_nomina_pct, :margen_neto_pct, :margen_operacional_pct,
                    :part_cafe_pct, :part_leche_pct, :part_frutas_pct,
                    :part_reposteria_pct, :part_otros_costos_pct
                  )";
                $stmt = $pdo->prepare($sql);
                $stmt->execute(array_merge($datos, $resultados));
                $id = (int)$pdo->lastInsertId();
            }
        } catch (PDOException $e) {
            $errores[] = 'Error al guardar en BD: ' . $e->getMessage();
        }
    }
}

function m($n) { return '$' . number_format($n, 0, ',', '.'); }
function p($n) { return number_format($n, 2, ',', '.') . '%'; }

function indicador_ok($valor, $meta, $tipo = 'max') {
    if ($tipo === 'max') {
        return $valor <= $meta;
    }
    return $valor >= $meta;
}

function badgeIndicador($valor, $meta, $tipo = 'max') {
    $ok = indicador_ok($valor, $meta, $tipo);
    $class = $ok ? 'badge-bien' : 'badge-critico';
    return '<span class="badge ' . $class . '">' . p($valor) . '</span>';
}

$hayResultados = ($_SERVER['REQUEST_METHOD'] === 'POST' && empty($errores));
?>
<!DOCTYPE html>
<html lang="es">
<head>
  <meta charset="UTF-8">
  <title>Calculadora de Proyecciones - Nativos</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.1/dist/chart.umd.min.js"></script>
  <style>
    body {
      background: #f5f7fb;
    }
    .card {
      border-radius: 16px;
      box-shadow: 0 10px 25px rgba(0,0,0,0.05);
      border: none;
    }
    .section-title {
      border-left: 4px solid #28a745;
      padding-left: 10px;
      font-weight: 600;
      margin-top: 20px;
      margin-bottom: 10px;
    }
    .badge-critico { background:#dc3545; }
    .badge-bien { background:#28a745; }
    .badge-alerta { background:#ffc107; color:#000; }

    @media print {
      body { background: #fff; }
      .no-print { display:none !important; }
      .card { box-shadow:none; border:none; }
      .container { max-width: 100%; }
    }
  </style>
</head>
<body>
<div class="container py-4">

  <nav class="navbar navbar-expand-lg navbar-light bg-white rounded shadow-sm mb-4 no-print">
    <div class="container-fluid">
      <a class="navbar-brand fw-semibold" href="index.php">Calculadora de Proyecciones</a>
      <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navMain">
        <span class="navbar-toggler-icon"></span>
      </button>
      <div class="collapse navbar-collapse" id="navMain">
        <ul class="navbar-nav me-auto mb-2 mb-lg-0">
          <li class="nav-item">
            <a class="nav-link active" aria-current="page" href="index.php">Nuevo / Editar reporte</a>
          </li>
          <li class="nav-item">
            <a class="nav-link" href="reportes.php">Listado de reportes</a>
          </li>
          <li class="nav-item">
            <a class="nav-link" href="analisis.php">Análisis anual</a>
          </li>
        </ul>
        <span class="navbar-text me-3 small">
          <?= htmlspecialchars($_SESSION['user_name'] ?? $_SESSION['user'] ?? '') ?>
        </span>
        <a href="logout.php" class="btn btn-outline-danger btn-sm">Cerrar sesión</a>
      </div>
    </div>
  </nav>

  <div class="d-flex justify-content-between align-items-center mb-3 no-print">
    <div>
      <h1 class="h4 mb-0"><?= $id > 0 ? 'Editar reporte' : 'Nuevo reporte' ?></h1>
    </div>
    <button class="btn btn-outline-secondary" onclick="window.print()">Imprimir / Guardar en PDF</button>
  </div>

  <?php if (!empty($errores)): ?>
    <div class="alert alert-danger">
      <ul class="mb-0">
        <?php foreach ($errores as $e): ?>
          <li><?= htmlspecialchars($e) ?></li>
        <?php endforeach; ?>
      </ul>
    </div>
  <?php endif; ?>

  <!-- ===== FORMULARIO ===== -->
  <div class="card mb-4 no-print" id="formulario">
    <div class="card-body">
      <form method="post">
        <input type="hidden" name="id" value="<?= htmlspecialchars($id) ?>">
        <div class="section-title">Datos generales</div>
        <div class="row g-3">
          <div class="col-md-4">
            <label class="form-label">Nombre Franquicia Nativos *</label>
            <input type="text" name="franquicia" class="form-control" required
                   value="<?= htmlspecialchars($datos['franquicia'] ?? '') ?>">
          </div>
          <div class="col-md-4">
            <label class="form-label">Nombre Responsable *</label>
            <input type="text" name="responsable_nombre" class="form-control" required
                   value="<?= htmlspecialchars($datos['responsable_nombre'] ?? '') ?>">
          </div>
          <div class="col-md-4">
            <label class="form-label">Apellidos Responsable *</label>
            <input type="text" name="responsable_apellidos" class="form-control" required
                   value="<?= htmlspecialchars($datos['responsable_apellidos'] ?? '') ?>">
          </div>

          <div class="col-md-3">
            <label class="form-label">Departamento *</label>
            <input type="text" name="departamento" class="form-control" required
                   value="<?= htmlspecialchars($datos['departamento'] ?? '') ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">Ciudad *</label>
            <input type="text" name="ciudad" class="form-control" required
                   value="<?= htmlspecialchars($datos['ciudad'] ?? '') ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">Email *</label>
            <input type="email" name="email" class="form-control" required
                   value="<?= htmlspecialchars($datos['email'] ?? '') ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">Fecha de reporte *</label>
            <input type="date" name="fecha_reporte" class="form-control" required
                   value="<?= htmlspecialchars($datos['fecha_reporte'] ?? '') ?>">
          </div>

          <div class="col-md-4">
            <label class="form-label">Mes de reporte *</label>
            <select name="mes_analisis" class="form-select" required>
              <option value="">Selecciona el mes...</option>
              <?php $mesSel = $datos['mes_analisis'] ?? ''; ?>
              <?php foreach ($MESES_REPORTE as $m): ?>
                <option value="<?= htmlspecialchars($m) ?>" <?= $mesSel === $m ? 'selected' : '' ?>>
                  <?= htmlspecialchars($m) ?>
                </option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-md-4">
            <label class="form-label">Ventas del mes a analizar *</label>
            <input type="number" step="0.01" min="0" name="ventas_mes" class="form-control" required
                   value="<?= htmlspecialchars($datos['ventas_mes'] ?? '') ?>">
          </div>
          <div class="col-md-4">
            <label class="form-label">Número de empleados *</label>
            <input type="number" min="1" name="num_empleados" class="form-control" required
                   value="<?= htmlspecialchars($datos['num_empleados'] ?? '') ?>">
          </div>
        </div>

        <hr class="my-4">

        <div class="section-title">Gastos operacionales</div>
        <div class="row g-3">
          <div class="col-md-3">
            <label class="form-label">Arriendo + Administración *</label>
            <input type="number" step="0.01" min="0" name="arriendo_admin" class="form-control" required
                   value="<?= htmlspecialchars($datos['arriendo_admin'] ?? '') ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">Nómina (con SS) *</label>
            <input type="number" step="0.01" min="0" name="nomina" class="form-control" required
                   value="<?= htmlspecialchars($datos['nomina'] ?? '') ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">Servicios Públicos *</label>
            <input type="number" step="0.01" min="0" name="servicios_publicos" class="form-control" required
                   value="<?= htmlspecialchars($datos['servicios_publicos'] ?? '') ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">Contador *</label>
            <input type="number" step="0.01" min="0" name="contador" class="form-control" required
                   value="<?= htmlspecialchars($datos['contador'] ?? '') ?>">
          </div>

          <div class="col-md-3">
            <label class="form-label">Calidad *</label>
            <input type="number" step="0.01" min="0" name="calidad" class="form-control" required
                   value="<?= htmlspecialchars($datos['calidad'] ?? '') ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">Papelería *</label>
            <input type="number" step="0.01" min="0" name="papeleria" class="form-control" required
                   value="<?= htmlspecialchars($datos['papeleria'] ?? '') ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">Transporte y parqueadero *</label>
            <input type="number" step="0.01" min="0" name="transporte_parqueadero" class="form-control" required
                   value="<?= htmlspecialchars($datos['transporte_parqueadero'] ?? '') ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">Impuestos (Ind. y Com., Hacienda) *</label>
            <input type="number" step="0.01" min="0" name="impuestos" class="form-control" required
                   value="<?= htmlspecialchars($datos['impuestos'] ?? '') ?>">
          </div>

          <div class="col-md-3">
            <label class="form-label">Mantenimiento *</label>
            <input type="number" step="0.01" min="0" name="mantenimiento" class="form-control" required
                   value="<?= htmlspecialchars($datos['mantenimiento'] ?? '') ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">Mercadeo y activaciones *</label>
            <input type="number" step="0.01" min="0" name="mercadeo_activaciones" class="form-control" required
                   value="<?= htmlspecialchars($datos['mercadeo_activaciones'] ?? '') ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">Plataformas (POS y FE) *</label>
            <input type="number" step="0.01" min="0" name="plataformas" class="form-control" required
                   value="<?= htmlspecialchars($datos['plataformas'] ?? '') ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">Aseo (insumos y materiales) *</label>
            <input type="number" step="0.01" min="0" name="aseo_insumos" class="form-control" required
                   value="<?= htmlspecialchars($datos['aseo_insumos'] ?? '') ?>">
          </div>

          <div class="col-md-3">
            <label class="form-label">Otros gastos *</label>
            <input type="number" step="0.01" min="0" name="otros_gastos" class="form-control" required
                   value="<?= htmlspecialchars($datos['otros_gastos'] ?? '') ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">Crédito *</label>
            <input type="number" step="0.01" min="0" name="credito" class="form-control" required
                   value="<?= htmlspecialchars($datos['credito'] ?? '') ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">Gastos legales *</label>
            <input type="number" step="0.01" min="0" name="gastos_legales" class="form-control" required
                   value="<?= htmlspecialchars($datos['gastos_legales'] ?? '') ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">Gastos financieros *</label>
            <input type="number" step="0.01" min="0" name="gastos_financieros" class="form-control" required
                   value="<?= htmlspecialchars($datos['gastos_financieros'] ?? '') ?>">
          </div>
        </div>

        <hr class="my-4">

        <div class="section-title">Costos de producción</div>
        <div class="row g-3">
          <div class="col-md-2">
            <label class="form-label">Café *</label>
            <input type="number" step="0.01" min="0" name="cafe" class="form-control" required
                   value="<?= htmlspecialchars($datos['cafe'] ?? '') ?>">
          </div>
          <div class="col-md-2">
            <label class="form-label">Leche *</label>
            <input type="number" step="0.01" min="0" name="leche" class="form-control" required
                   value="<?= htmlspecialchars($datos['leche'] ?? '') ?>">
          </div>
          <div class="col-md-2">
            <label class="form-label">Frutas y verduras *</label>
            <input type="number" step="0.01" min="0" name="frutas_verduras" class="form-control" required
                   value="<?= htmlspecialchars($datos['frutas_verduras'] ?? '') ?>">
          </div>
          <div class="col-md-2">
            <label class="form-label">Repostería *</label>
            <input type="number" step="0.01" min="0" name="reposteria" class="form-control" required
                   value="<?= htmlspecialchars($datos['reposteria'] ?? '') ?>">
          </div>
          <div class="col-md-2">
            <label class="form-label">Otros costos *</label>
            <input type="number" step="0.01" min="0" name="otros_costos" class="form-control" required
                   value="<?= htmlspecialchars($datos['otros_costos'] ?? '') ?>">
          </div>
          <div class="col-md-2">
            <label class="form-label">Inventario actual *</label>
            <input type="number" step="0.01" min="0" name="inventario" class="form-control" required
                   value="<?= htmlspecialchars($datos['inventario'] ?? '') ?>">
          </div>
        </div>

        <div class="mt-4 text-end">
          <button type="submit" class="btn btn-success">
            <?= $id > 0 ? 'Actualizar reporte' : 'Calcular y guardar reporte' ?>
          </button>
        </div>
      </form>
    </div>
  </div>

  <!-- ===== REPORTE ===== -->
  <?php if ($hayResultados): ?>
    <div class="card mb-4" id="reporte">
      <div class="card-body">
        <h2 class="h4 mb-3">Informe de Proyección - <?= htmlspecialchars($datos['franquicia']) ?></h2>
        <p class="text-muted mb-1">
          Fecha de reporte: <strong><?= fechaBonita($datos['fecha_reporte'] ?? '') ?></strong>
        </p>
        <p class="text-muted mb-1">
          Mes de análisis: <strong><?= htmlspecialchars($datos['mes_analisis'] ?: 'No especificado') ?></strong>
        </p>
        <p class="text-muted mb-4">
          Departamento: <strong><?= htmlspecialchars($datos['departamento'] ?? '') ?></strong> ·
          Ciudad: <strong><?= htmlspecialchars($datos['ciudad'] ?? '') ?></strong><br>
          Email de contacto: <strong><?= htmlspecialchars($datos['email'] ?? '') ?></strong>
        </p>

        <div class="row g-3 mb-4">
          <div class="col-md-3">
            <div class="p-3 bg-light rounded">
              <div class="text-muted small">Ventas del mes</div>
              <div class="fw-bold fs-5"><?= m($datos['ventas_mes']) ?></div>
            </div>
          </div>
          <div class="col-md-3">
            <div class="p-3 bg-light rounded">
              <div class="text-muted small">Promedio ventas día</div>
              <div class="fw-bold fs-5"><?= m($resultados['promedio_dia']) ?></div>
            </div>
          </div>
          <div class="col-md-3">
            <div class="p-3 bg-light rounded">
              <div class="text-muted small">Total costos</div>
              <div class="fw-bold fs-5"><?= m($resultados['total_costos']) ?></div>
              <div class="small">
                Costo producción:
                <?= badgeIndicador($resultados['costo_produccion_pct'], 30, 'max') ?>
                <span class="ms-1 small text-muted">(meta ≤ 30%)</span>
              </div>
            </div>
          </div>
          <div class="col-md-3">
            <div class="p-3 bg-light rounded">
              <div class="text-muted small">Total gastos (operacionales)</div>
              <div class="fw-bold fs-5"><?= m($resultados['total_gastos']) ?></div>
            </div>
          </div>
        </div>

        <div class="row g-3 mb-4">
          <div class="col-md-3">
            <div class="p-3 bg-light rounded">
              <div class="text-muted small">Utilidad bruta</div>
              <div class="fw-bold fs-5"><?= m($resultados['utilidad_bruta']) ?></div>
            </div>
          </div>
          <div class="col-md-3">
            <div class="p-3 bg-light rounded">
              <div class="text-muted small">Utilidad operacional</div>
              <div class="fw-bold fs-5"><?= m($resultados['utilidad_operacional']) ?></div>
              <span class="mt-1 d-inline-block">
                Margen operacional:
                <?= badgeIndicador($resultados['margen_operacional_pct'], 18, 'min') ?>
                <span class="ms-1 small text-muted">(meta ≥ 18%)</span>
              </span>
            </div>
          </div>
          <div class="col-md-3">
            <div class="p-3 bg-light rounded">
              <div class="text-muted small">Utilidad neta</div>
              <div class="fw-bold fs-5"><?= m($resultados['utilidad_neta']) ?></div>
              <span class="mt-1 d-inline-block">
                Margen neto:
                <?= badgeIndicador($resultados['margen_neto_pct'], 16, 'min') ?>
                <span class="ms-1 small text-muted">(meta ≥ 16%)</span>
              </span>
            </div>
          </div>
          <div class="col-md-3">
            <div class="p-3 bg-light rounded">
              <div class="text-muted small">Utilidad neta - inventario</div>
              <div class="fw-bold fs-5"><?= m($resultados['utilidad_neta_menos_inventario']) ?></div>
              <div class="small">
                Sobre ventas:
                <strong><?= p($resultados['utilidad_neta_menos_inv_sobre_ventas']) ?></strong>
              </div>
            </div>
          </div>
        </div>

        <div class="mb-4">
          <div class="section-title">Inventario</div>
          <p>
            Inventario actual: <strong><?= m($datos['inventario']) ?></strong><br>
            Inventario sugerido (10% de las ventas): <strong><?= m($resultados['inventario_sugerido']) ?></strong>
          </p>
        </div>

        <div class="mb-4">
          <div class="section-title">Resultados esperados</div>
          <div class="table-responsive">
            <table class="table table-sm align-middle">
              <thead>
                <tr>
                  <th>Indicador</th>
                  <th>Meta</th>
                  <th>Resultado</th>
                  <th>Estado</th>
                </tr>
              </thead>
              <tbody>
                <?php
                  $indicadores = [
                    [
                      'label' => 'Costo de producción',
                      'meta'  => '≤ 30%',
                      'valor' => $resultados['costo_produccion_pct'],
                      'tipo'  => 'max',
                      'limiteNum' => 30,
                    ],
                    [
                      'label' => 'Participación arriendo',
                      'meta'  => '≤ 20%',
                      'valor' => $resultados['part_arriendo_pct'],
                      'tipo'  => 'max',
                      'limiteNum' => 20,
                    ],
                    [
                      'label' => 'Participación servicios públicos',
                      'meta'  => '≤ 4%',
                      'valor' => $resultados['part_servicios_pct'],
                      'tipo'  => 'max',
                      'limiteNum' => 4,
                    ],
                    [
                      'label' => 'Participación nómina',
                      'meta'  => '≤ 22%',
                      'valor' => $resultados['part_nomina_pct'],
                      'tipo'  => 'max',
                      'limiteNum' => 22,
                    ],
                    [
                      'label' => 'Margen neto',
                      'meta'  => '≥ 16%',
                      'valor' => $resultados['margen_neto_pct'],
                      'tipo'  => 'min',
                      'limiteNum' => 16,
                    ],
                    [
                      'label' => 'Margen operacional',
                      'meta'  => '≥ 18%',
                      'valor' => $resultados['margen_operacional_pct'],
                      'tipo'  => 'min',
                      'limiteNum' => 18,
                    ],
                    [
                      'label' => 'Participación café',
                      'meta'  => '≤ 35%',
                      'valor' => $resultados['part_cafe_pct'],
                      'tipo'  => 'max',
                      'limiteNum' => 35,
                    ],
                    [
                      'label' => 'Participación leche',
                      'meta'  => '≤ 15%',
                      'valor' => $resultados['part_leche_pct'],
                      'tipo'  => 'max',
                      'limiteNum' => 15,
                    ],
                    [
                      'label' => 'Participación frutas y verduras',
                      'meta'  => '≤ 20%',
                      'valor' => $resultados['part_frutas_pct'],
                      'tipo'  => 'max',
                      'limiteNum' => 20,
                    ],
                    [
                      'label' => 'Participación repostería',
                      'meta'  => '≤ 15%',
                      'valor' => $resultados['part_reposteria_pct'],
                      'tipo'  => 'max',
                      'limiteNum' => 15,
                    ],
                    [
                      'label' => 'Participación otros costos',
                      'meta'  => '≤ 15%',
                      'valor' => $resultados['part_otros_costos_pct'],
                      'tipo'  => 'max',
                      'limiteNum' => 15,
                    ],
                  ];
                ?>
                <?php foreach ($indicadores as $ind):
                  $ok = indicador_ok($ind['valor'], $ind['limiteNum'], $ind['tipo']);
                ?>
                  <tr>
                    <td><?= htmlspecialchars($ind['label']) ?></td>
                    <td><?= htmlspecialchars($ind['meta']) ?></td>
                    <td><?= badgeIndicador($ind['valor'], $ind['limiteNum'], $ind['tipo']) ?></td>
                    <td>
                      <?php if ($ok): ?>
                        <span class="text-success fw-semibold">OK</span>
                      <?php else: ?>
                        <span class="text-danger fw-semibold">Revisar</span>
                      <?php endif; ?>
                    </td>
                  </tr>
                <?php endforeach; ?>
              </tbody>
            </table>
          </div>
        </div>

        <div class="row g-4 mb-4">
          <div class="col-md-6">
            <div class="section-title">Resumen financiero</div>
            <canvas id="chartResumen"></canvas>
          </div>
          <div class="col-md-6">
            <div class="section-title">Distribución de costos</div>
            <canvas id="chartCostos"></canvas>
          </div>
        </div>

        <div class="mt-4">
          <div class="section-title">Informe detallado</div>
          <p>
            Un cordial saludo. <strong><?= htmlspecialchars($datos['franquicia']) ?></strong>
          </p>
          <p>
            Fecha de reporte: <strong><?= fechaBonita($datos['fecha_reporte'] ?? '') ?></strong><br>
            Ubicación: <strong><?= htmlspecialchars($datos['ciudad'] . ', ' . $datos['departamento']) ?></strong><br>
            Contacto: <strong><?= htmlspecialchars($datos['email']) ?></strong>
          </p>
          <p>
            Espero que todo esté bien.
            <strong><?= htmlspecialchars($datos['responsable_nombre'] . ' ' . $datos['responsable_apellidos']) ?></strong>
          </p>
          <p>
            Gracias por cumplir con el compromiso del envío del informe mensual de gastos, costos y ventas.
            Esto nos permitirá crecer juntos.
          </p>

          <ol>
            <li>
              <strong>Ventas (<?= htmlspecialchars($datos['franquicia']) ?>)</strong><br>
              Ventas (<?= htmlspecialchars($datos['mes_analisis'] ?: 'Mes analizado') ?>): <?= m($datos['ventas_mes']) ?><br>
              Promedio/día (<?= htmlspecialchars($datos['mes_analisis'] ?: 'Mes analizado') ?>):
              <?= m($resultados['promedio_dia']) ?>
            </li>
            <br>
            <li>
              <strong>Costos de producción</strong><br>
              Costos (<?= htmlspecialchars($datos['mes_analisis'] ?: 'Mes analizado') ?>): <?= m($resultados['total_costos']) ?>
              (<?= p($resultados['porcentaje_costos']) ?>)<br>
              No pueden superar el 30% y están en el <?= p($resultados['porcentaje_costos']) ?>.
            </li>
            <br>
            <li>
              <strong>Inventario</strong><br>
              Debería estar en <?= m($resultados['inventario_sugerido']) ?>.
              El inventario está en <?= m($datos['inventario']) ?>.
            </li>
            <br>
            <li>
              <strong>Gastos operacionales (<?= htmlspecialchars($datos['mes_analisis'] ?: 'Mes analizado') ?>)</strong><br>
              Gastos: <?= m($resultados['total_gastos']) ?>
            </li>
          </ol>

          <hr>

          <h5>Estado de Resultados (<?= htmlspecialchars($datos['mes_analisis'] ?: 'Mes analizado') ?>)</h5>
          <p>
            Ventas: <?= m($datos['ventas_mes']) ?><br>
            Costos: <?= m($resultados['total_costos']) ?> (<?= p($resultados['porcentaje_costos']) ?>)<br>
            Inventario: <?= m($datos['inventario']) ?><br>
            Utilidad bruta: <?= m($resultados['utilidad_bruta']) ?><br><br>

            Gastos: <?= m($resultados['total_gastos']) ?><br>
            Utilidad operacional: <?= m($resultados['utilidad_operacional']) ?><br>
            Margen operacional: <?= p($resultados['margen_operacional_pct']) ?><br><br>

            Menos crédito: <?= m($datos['credito']) ?><br>
            Menos gastos financieros: <?= m($datos['gastos_financieros']) ?><br>
            Menos impuestos: <?= m($datos['impuestos']) ?><br><br>

            Utilidad neta: <?= m($resultados['utilidad_neta']) ?><br>
            Margen neto: <?= p($resultados['margen_neto_pct']) ?><br><br>

            La utilidad neta con el actual costo del <?= p($resultados['porcentaje_costos']) ?>,
            luego de cancelar gastos financieros, crédito y tributarios y sin inventario:
            <?= m($datos['inventario']) ?>, es
            <?= m($resultados['utilidad_neta_menos_inventario']) ?>,
            que representa el <?= p($resultados['utilidad_neta_menos_inv_sobre_ventas']) ?>.
            Esta fue la ganancia real percibida por ustedes.
          </p>

          <p class="mt-3">
            Gracias.
          </p>
        </div>

      </div>
    </div>
  <?php endif; ?>

</div>

<?php if ($hayResultados): ?>
<script>
  const ventas    = <?= json_encode($datos['ventas_mes'] ?? 0) ?>;
  const totalCostos = <?= json_encode($resultados['total_costos'] ?? 0) ?>;
  const totalGastos = <?= json_encode($resultados['total_gastos'] ?? 0) ?>;
  const uBruta   = <?= json_encode($resultados['utilidad_bruta'] ?? 0) ?>;
  const uOper    = <?= json_encode($resultados['utilidad_operacional'] ?? 0) ?>;
  const uNeta    = <?= json_encode($resultados['utilidad_neta'] ?? 0) ?>;

  const costosDetalle = {
    cafe: <?= json_encode($datos['cafe'] ?? 0) ?>,
    leche: <?= json_encode($datos['leche'] ?? 0) ?>,
    frutas: <?= json_encode($datos['frutas_verduras'] ?? 0) ?>,
    reposteria: <?= json_encode($datos['reposteria'] ?? 0) ?>,
    otros: <?= json_encode($datos['otros_costos'] ?? 0) ?>
  };

  const ctxRes = document.getElementById('chartResumen');
  new Chart(ctxRes, {
    type: 'bar',
    data: {
      labels: ['Ventas', 'Total Costos', 'Total Gastos', 'Utilidad Bruta', 'Utilidad Operacional', 'Utilidad Neta'],
      datasets: [{
        label: 'Valor ($)',
        data: [ventas, totalCostos, totalGastos, uBruta, uOper, uNeta]
      }]
    },
    options: {
      responsive: true,
      plugins: {
        legend: { display: false },
        tooltip: {
          callbacks: {
            label: function(ctx) {
              const v = ctx.parsed.y || 0;
              return ' $' + v.toLocaleString('es-CO');
            }
          }
        }
      },
      scales: {
        y: {
          beginAtZero: true
        }
      }
    }
  });

  const ctxCostos = document.getElementById('chartCostos');
  new Chart(ctxCostos, {
    type: 'pie',
    data: {
      labels: ['Café', 'Leche', 'Frutas y verduras', 'Repostería', 'Otros'],
      datasets: [{
        data: [
          costosDetalle.cafe,
          costosDetalle.leche,
          costosDetalle.frutas,
          costosDetalle.reposteria,
          costosDetalle.otros
        ]
      }]
    },
    options: {
      responsive: true,
      plugins: {
        tooltip: {
          callbacks: {
            label: function(ctx) {
              const total = ctx.dataset.data.reduce((a, b) => a + b, 0);
              const val = ctx.parsed || 0;
              const pct = total > 0 ? (val / total * 100) : 0;
              return ctx.label + ': $' + val.toLocaleString('es-CO') +
                ' (' + pct.toFixed(1) + '%)';
            }
          }
        }
      }
    }
  });
</script>
<?php endif; ?>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
