<?php
require_once __DIR__ . '/auth.php';
require_login();
global $pdo;

// Años disponibles
$dbError = '';

$years = [];
try {
    $stmtYears = $pdo->query("SELECT DISTINCT YEAR(fecha_reporte) AS anio FROM proyecciones ORDER BY anio DESC");
    $years = $stmtYears->fetchAll(PDO::FETCH_COLUMN);
} catch (PDOException $e) {
    $dbError = 'Error consultando años: '.$e->getMessage();
}

$anioSel = $years ? (int)$years[0] : (int)date('Y');
if (isset($_GET['anio']) && ctype_digit($_GET['anio'])) {
    $anioSel = (int)$_GET['anio'];
}

// Franquicias disponibles
$franquicias = [];
try {
    $stmtFran = $pdo->query("SELECT DISTINCT franquicia FROM proyecciones ORDER BY franquicia ASC");
    $franquicias = $stmtFran->fetchAll(PDO::FETCH_COLUMN);
} catch (PDOException $e) {
    $dbError = $dbError ?: 'Error consultando franquicias: '.$e->getMessage();
}

$franSel = $franquicias[0] ?? '';
if (isset($_GET['franquicia']) && $_GET['franquicia'] !== '') {
    $franSel = $_GET['franquicia'];
}

$rows = [];
if ($years && $franquicias && !$dbError) {
    try {
        $sql = "SELECT
                  mes_analisis,
                  fecha_reporte,
                  ventas_mes,
                  total_costos,
                  total_gastos,
                  utilidad_neta
                FROM proyecciones
                WHERE YEAR(fecha_reporte) = :anio
                  AND franquicia = :fran
                ORDER BY fecha_reporte ASC";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':anio' => $anioSel,
            ':fran' => $franSel,
        ]);
        $rows = $stmt->fetchAll();
    } catch (PDOException $e) {
        $dbError = 'Error consultando datos del análisis: '.$e->getMessage();
    }
}

function m($n) { return '$' . number_format($n, 0, ',', '.'); }
function p($n) { return number_format($n, 2, ',', '.') . '%'; }
function fechaBonita($fecha) {
    if (!$fecha) return '';
    try {
        $dt = new DateTime($fecha);
        return $dt->format('d/m/Y');
    } catch (Exception $e) {
        return htmlspecialchars($fecha);
    }
}

// Preparar datos para Chart.js
$labels = [];
$dataVentas = [];
$dataUtilNeta = [];
foreach ($rows as $r) {
    $labels[]      = $r['mes_analisis'];
    $dataVentas[]  = (float)$r['ventas_mes'];
    $dataUtilNeta[] = (float)$r['utilidad_neta'];
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
  <meta charset="UTF-8">
  <title>Rituales - Análisis anual</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.1/dist/chart.umd.min.js"></script>
  <style>
  :root {
    --rituales-primary: #5a3825;
    --rituales-primary-dark: #3d2518;
    --rituales-accent: #c58b57;
    --rituales-bg: #f7f1ea;
  }
  body {
    background: var(--rituales-bg);
    font-family: system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif;
  }
  .card {
    border-radius: 16px;
    box-shadow: 0 10px 25px rgba(0,0,0,0.05);
    border: none;
  }
  .navbar-rituales {
    background: var(--rituales-primary);
  }
  .navbar-rituales .navbar-brand,
  .navbar-rituales .nav-link,
  .navbar-rituales .navbar-text {
    color: #ffffff !important;
  }
  .navbar-rituales .nav-link.active {
    font-weight: 600;
    text-decoration: underline;
    text-decoration-thickness: 2px;
    text-underline-offset: 4px;
  }
  .navbar-rituales .navbar-toggler {
    border-color: rgba(255,255,255,0.6);
  }
  .navbar-rituales .navbar-toggler-icon {
    filter: invert(1);
  }
  .btn-rituales {
    background: var(--rituales-primary);
    border-color: var(--rituales-primary);
    color: #fff;
  }
  .btn-rituales:hover {
    background: var(--rituales-primary-dark);
    border-color: var(--rituales-primary-dark);
    color: #fff;
  }
  .section-title {
    border-left: 4px solid var(--rituales-accent);
    padding-left: 10px;
    font-weight: 600;
    margin-top: 20px;
    margin-bottom: 10px;
    color: #5a3825;
  }
  .badge-critico { background:#dc3545; }
  .badge-bien { background:#28a745; }
  .badge-alerta { background:#ffc107; color:#000; }
  @media print {
    body { background:#fff; }
    .no-print { display:none !important; }
    .card { box-shadow:none; border:none; }
    .container { max-width: 100%; }
  }
  </style>
</head>
<body>
<div class="container py-4">

  <nav class="navbar navbar-expand-lg navbar-dark navbar-rituales rounded shadow-sm mb-4 no-print">
    <div class="container-fluid">
      <a class="navbar-brand d-flex align-items-center" href="index.php">
        <img src="logorituales.png" alt="Rituales" class="me-2" style="height:32px; width:auto;">
        <span>Rituales · Calculadora de Proyecciones</span>
      </a>
      <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navMain">
        <span class="navbar-toggler-icon"></span>
      </button>
      <div class="collapse navbar-collapse" id="navMain">
        <ul class="navbar-nav me-auto mb-2 mb-lg-0">
          <li class="nav-item">
            <a class="nav-link" href="index.php">Nuevo / Editar reporte</a>
          </li>
          <li class="nav-item">
            <a class="nav-link" href="reportes.php">Listado de reportes</a>
          </li>
          <li class="nav-item">
            <a class="nav-link active" aria-current="page" href="analisis.php">Análisis anual</a>
          </li>
        </ul>
        <span class="navbar-text me-3 small">
          <?= htmlspecialchars($_SESSION['user_name'] ?? $_SESSION['user'] ?? '') ?>
        </span>
        <a href="logout.php" class="btn btn-outline-light btn-sm">Cerrar sesión</a>
      </div>
    </div>
  </nav>

  <div class="card mb-4">
    <div class="card-body">
      <div class="d-flex justify-content-between align-items-center mb-3">
        <div>
          <h1 class="h5 mb-0">Análisis comparativo anual</h1>
          <small class="text-muted">Evolución de ventas y utilidad neta por mes</small>
        </div>
        <button class="btn btn-outline-secondary btn-sm no-print" onclick="window.print()">Imprimir / PDF</button>
      </div>

      <?php if ($dbError): ?>
        <div class="alert alert-danger">
          <strong>Error de base de datos:</strong><br>
          <span class="small"><?= htmlspecialchars($dbError) ?></span>
        </div>
      <?php endif; ?>

      <?php if (!$years || !$franquicias): ?>
        <p class="text-muted">Aún no hay información suficiente para el análisis anual.</p>
      <?php else: ?>
        <form method="get" class="row g-3 align-items-end mb-3 no-print">
          <div class="col-md-3">
            <label class="form-label">Año</label>
            <select name="anio" class="form-select">
              <?php foreach ($years as $y): ?>
                <option value="<?= (int)$y ?>" <?= (int)$y === (int)$anioSel ? 'selected' : '' ?>>
                  <?= (int)$y ?>
                </option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-md-5">
            <label class="form-label">Franquicia</label>
            <select name="franquicia" class="form-select">
              <?php foreach ($franquicias as $f): ?>
                <option value="<?= htmlspecialchars($f) ?>" <?= $f === $franSel ? 'selected' : '' ?>>
                  <?= htmlspecialchars($f) ?>
                </option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-md-2">
            <button type="submit" class="btn btn-rituales w-100">Ver análisis</button>
          </div>
        </form>

        <p class="mb-3">
          Franquicia: <strong><?= htmlspecialchars($franSel) ?></strong><br>
          Año: <strong><?= (int)$anioSel ?></strong>
        </p>

        <?php if (!$rows): ?>
          <p class="text-muted">No hay reportes para esta combinación de año y franquicia.</p>
        <?php else: ?>
          <div class="row g-4 mb-4">
            <div class="col-md-12">
              <div class="section-title">Evolución de ventas y utilidad neta</div>
              <canvas id="chartAnual"></canvas>
            </div>
          </div>

          <div class="section-title">Detalle mensual</div>
          <div class="table-responsive">
            <table class="table table-sm align-middle">
              <thead>
                <tr>
                  <th>Fecha</th>
                  <th>Mes</th>
                  <th>Ventas</th>
                  <th>Total costos</th>
                  <th>Total gastos</th>
                  <th>Utilidad neta</th>
                  <th>Margen neto</th>
                </tr>
              </thead>
              <tbody>
                <?php foreach ($rows as $r): ?>
                  <?php
                    $ventas = (float)$r['ventas_mes'];
                    $uNeta  = (float)$r['utilidad_neta'];
                    $margenNeto = $ventas > 0 ? ($uNeta / $ventas * 100) : 0;
                  ?>
                  <tr>
                    <td><?= fechaBonita($r['fecha_reporte']) ?></td>
                    <td><?= htmlspecialchars($r['mes_analisis']) ?></td>
                    <td><?= m($r['ventas_mes']) ?></td>
                    <td><?= m($r['total_costos']) ?></td>
                    <td><?= m($r['total_gastos']) ?></td>
                    <td><?= m($r['utilidad_neta']) ?></td>
                    <td><?= p($margenNeto) ?></td>
                  </tr>
                <?php endforeach; ?>
              </tbody>
            </table>
          </div>
        <?php endif; ?>
      <?php endif; ?>
    </div>
  </div>

</div>

<?php if ($rows): ?>
<script>
  const labels      = <?= json_encode($labels) ?>;
  const dataVentas  = <?= json_encode($dataVentas) ?>;
  const dataUtilidad = <?= json_encode($dataUtilNeta) ?>;

  const ctx = document.getElementById('chartAnual');
  new Chart(ctx, {
    type: 'bar',
    data: {
      labels: labels,
      datasets: [
        {
          label: 'Ventas',
          data: dataVentas
        },
        {
          label: 'Utilidad neta',
          data: dataUtilidad
        }
      ]
    },
    options: {
      responsive: true,
      plugins: {
        tooltip: {
          callbacks: {
            label: function(ctx) {
              const v = ctx.parsed.y || 0;
              return ctx.dataset.label + ': $' + v.toLocaleString('es-CO');
            }
          }
        }
      },
      scales: {
        y: {
          beginAtZero: true
        }
      }
    }
  });
</script>
<?php endif; ?>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
